<?php
// criar_pagamento.php

require __DIR__ . '/config.php';
require __DIR__ . '/db.php';
require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/functions.php';

use MercadoPago\MercadoPagoConfig;
use MercadoPago\Client\Preference\PreferenceClient;
use MercadoPago\Exceptions\MPApiException;

/**
 * Função auxiliar pra mostrar erro amigável e parar execução.
 */
function erro_pagamento($msg) {
    header('Content-Type: text/html; charset=utf-8');
    echo "<h2>Erro ao iniciar pagamento</h2>";
    echo "<p>{$msg}</p>";
    echo '<p><a href=\"pagamento.php\">Voltar</a></p>';
    exit;
}

// Dados do comprador vindos do formulário
$nome   = isset($_POST['nome']) ? trim($_POST['nome']) : '';
$email  = isset($_POST['email']) ? trim($_POST['email']) : '';
$whats  = isset($_POST['whatsapp']) ? trim($_POST['whatsapp']) : '';
$tipo   = isset($_POST['tipo']) ? trim($_POST['tipo']) : '';
$doc    = isset($_POST['doc']) ? trim($_POST['doc']) : '';

// Valor do APP: busca no DB, fallback 29.90
$appPriceStr = get_app_setting($pdo, 'app_price', '29.90');
$precoPadrao = (float)$appPriceStr;
$preco       = isset($_POST['preco']) ? floatval($_POST['preco']) : $precoPadrao;
if ($preco <= 0) {
    $preco = $precoPadrao;
}

// Gateway padrão pelo banco
$gateway = get_default_gateway($pdo);
if (!$gateway) {
    erro_pagamento('Nenhum gateway de pagamento ativo e padrão configurado. Ajuste no painel Admin → Gateways.');
}

$slug   = $gateway['slug'];
$config = $gateway['config'] ?? [];

// -----------------------------------------------------
// ROTAS POR GATEWAY
// -----------------------------------------------------

switch ($slug) {

    // =================================================
    // MERCADO PAGO
    // =================================================
    case 'mercadopago':
        $accessToken = $config['access_token'] ?? '';
        if (empty($accessToken)) {
            erro_pagamento('Access Token do Mercado Pago não configurado. Verifique em Admin → Gateways (Mercado Pago).');
        }

        MercadoPagoConfig::setAccessToken($accessToken);

        $baseUrl = $config['base_url'] ?? '';

        try {
            $item = [
                "id"          => "licenca_app_preco",
                "title"       => "Licença App Formação de Preço de Venda",
                "description" => "Acesso ao aplicativo de formação de preço de venda.",
                "currency_id" => "BRL",
                "quantity"    => 1,
                "unit_price"  => $preco
            ];

            $payer = [];
            if ($email) $payer["email"] = $email;
            if ($nome)  $payer["name"]  = $nome;

            $request = [
                "items" => [$item],
                "payer" => $payer,
                "metadata" => [
                    "nome"      => $nome,
                    "email"     => $email,
                    "whatsapp"  => $whats,
                    "tipo"      => $tipo,
                    "doc"       => $doc,
                    "origem"    => "app_formacao_preco"
                ]
            ];

            if (!empty($baseUrl) && substr($baseUrl, 0, 8) === 'https://') {
                $request["back_urls"] = [
                    "success" => $baseUrl . "/sucesso.php",
                    "failure" => $baseUrl . "/erro.php",
                    "pending" => $baseUrl . "/pendente.php",
                ];
                $request["auto_return"] = "approved";
            }

            $client     = new PreferenceClient();
            $preference = $client->create($request);

            if (!empty($preference->init_point)) {
                header('Location: ' . $preference->init_point);
                exit;
            } else {
                throw new Exception('Não foi possível obter o link de pagamento (init_point vazio).');
            }

        } catch (MPApiException $e) {
            erro_pagamento('Erro na API do Mercado Pago: ' . $e->getMessage());
        } catch (Exception $e) {
            erro_pagamento('Erro ao criar pagamento no Mercado Pago: ' . $e->getMessage());
        }
        break;

    // =================================================
    // PAGAR.ME - via link de checkout
    // =================================================
    case 'pagarme':
        $checkoutUrl = $config['checkout_url'] ?? '';
        if (empty($checkoutUrl)) {
            erro_pagamento('URL de checkout do Pagar.me não configurada. Defina em Admin → Gateways (Pagar.me).');
        }

        $url    = $checkoutUrl;
        $params = [];
        if ($email) $params['email'] = $email;
        if ($nome)  $params['name']  = $nome;

        if ($params) {
            $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($params);
        }

        header('Location: ' . $url);
        exit;

    // =================================================
    // ASAAS - via API (criação de cobrança e redirect invoiceUrl)
    // =================================================
    case 'asaas':

        // Lê configs do BD
        $apiKey      = $config['api_key']  ?? '';
        $configBase  = strtolower(trim($config['base_url'] ?? ''));

        // Decide ambiente com base no que vier em base_url:
        // - se contiver "sandbox" -> usa sandbox
        // - caso contrário -> produção
        if (strpos($configBase, 'sandbox') !== false) {
            $apiHost = 'https://api-sandbox.asaas.com';
        } else {
            $apiHost = 'https://api.asaas.com';
        }

        if (empty($apiKey)) {
            erro_pagamento('API Key do Asaas não configurada. Defina em Admin → Gateways (gateway Asaas).');
        }

        // Normaliza host (sem barra final)
        $apiHost = rtrim($apiHost, '/'); // ex: https://api.asaas.com

        /**
         * Função auxiliar para chamar API Asaas.
         * IMPORTANTE:
         *   - Sempre monta:  {host}/v3{endpoint}
         *   - endpoint deve começar com "/..."
         * Exemplo final:
         *   GET  https://api.asaas.com/v3/customers?email=...
         *   POST https://api.asaas.com/v3/customers
         *   POST https://api.asaas.com/v3/payments
         */
        $asaasRequest = function(string $method, string $endpoint, array $body = null) use ($apiKey, $apiHost) {
            // garante que endpoint começa com "/"
            if ($endpoint === '' || $endpoint[0] !== '/') {
                $endpoint = '/' . ltrim($endpoint, '/');
            }

            $url = $apiHost . '/v3' . $endpoint;   // <- AQUI é o formato oficial

            $ch = curl_init($url);
            $headers = [
                'Content-Type: application/json',
                'access_token: ' . $apiKey,
            ];

            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, strtoupper($method));

            if (!empty($body)) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
            }

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $err      = curl_error($ch);
            curl_close($ch);

            return [$httpCode, $response, $err, $url];
        };

        $customerId = null;

        // 1) Se TIVER e-mail, tenta buscar cliente pelo e-mail
        if (!empty($email)) {
            [$codeSearch, $respSearch, $errSearch, $urlSearch] =
                $asaasRequest('GET', '/customers?email=' . urlencode($email));

            if ($errSearch) {
                erro_pagamento(
                    'Erro ao comunicar com Asaas (busca cliente): ' .
                    htmlspecialchars($errSearch) .
                    '<br><small>URL: ' . htmlspecialchars($urlSearch) . '</small>'
                );
            }

            $dataSearch = json_decode($respSearch, true);
            if ($codeSearch < 400 && isset($dataSearch['data']) && is_array($dataSearch['data']) && count($dataSearch['data']) > 0) {
                $customerId = $dataSearch['data'][0]['id'] ?? null;
            }
        }

        // 2) Se NÃO encontrou cliente (ou não tinha e-mail), cria novo cliente
        if (!$customerId) {
            $nomeCliente = $nome ?: ($email ?: 'Cliente do App');

            $bodyCustomer = [
                'name' => $nomeCliente,
            ];

            if (!empty($email)) {
                $bodyCustomer['email'] = $email;
            }

            if (!empty($doc)) {
                $cpfCnpj = preg_replace('/\D+/', '', $doc);
                if ($cpfCnpj !== '') {
                    $bodyCustomer['cpfCnpj'] = $cpfCnpj;
                }
            }

            [$codeCust, $respCust, $errCust, $urlCust] =
                $asaasRequest('POST', '/customers', $bodyCustomer);

            if ($errCust) {
                erro_pagamento(
                    'Erro ao comunicar com Asaas (criação de cliente): ' .
                    htmlspecialchars($errCust) .
                    '<br><small>URL: ' . htmlspecialchars($urlCust) . '</small>'
                );
            }

            $dataCust = json_decode($respCust, true);

            if ($codeCust >= 400 || !is_array($dataCust) || !isset($dataCust['id'])) {
                $msgApi = '';
                if (is_array($dataCust) && isset($dataCust['errors'][0]['description'])) {
                    $msgApi = $dataCust['errors'][0]['description'];
                }

                $debug = 'URL: ' . $urlCust .
                         ' | HTTP ' . $codeCust .
                         ' | Resposta Asaas (customers): ' . htmlspecialchars($respCust);

                $msgFinal = 'Erro na criação de cliente Asaas: ' .
                            ($msgApi ?: 'Falha ao criar cliente no Asaas.') .
                            '<br><br><small>' . $debug . '</small>';

                erro_pagamento($msgFinal);
            }

            $customerId = $dataCust['id'] ?? null;
        }

        if (!$customerId) {
            erro_pagamento('Não foi possível obter o ID do cliente no Asaas (customerId vazio).');
        }

        // 3) Criar cobrança (payment)
        $bodyPayment = [
			'customer'    => $customerId,
			'billingType' => 'UNDEFINED', // você pode trocar para 'PIX', 'BOLETO', etc, se quiser
			'value'       => $preco,
			'description' => 'Licença App Formação de Preço de Venda',
			// dueDate obrigatório para esse tipo de cobrança
			// aqui estou colocando para 1 dia após hoje, formato YYYY-MM-DD
			'dueDate'     => date('Y-m-d', strtotime('+1 day')),
		];

        [$codePay, $respPay, $errPay, $urlPay] =
            $asaasRequest('POST', '/payments', $bodyPayment);

        if ($errPay) {
            erro_pagamento(
                'Erro ao comunicar com Asaas (criação de pagamento): ' .
                htmlspecialchars($errPay) .
                '<br><small>URL: ' . htmlspecialchars($urlPay) . '</small>'
            );
        }

        $dataPay = json_decode($respPay, true);

        if ($codePay >= 400 || !is_array($dataPay) || !isset($dataPay['id'])) {
            $msgApi = '';
            if (is_array($dataPay) && isset($dataPay['errors'][0]['description'])) {
                $msgApi = $dataPay['errors'][0]['description'];
            }

            $debug = 'URL: ' . $urlPay .
                     ' | HTTP ' . $codePay .
                     ' | Resposta Asaas (payments): ' . htmlspecialchars($respPay);

            $msgFinal = 'Erro na criação de cobrança Asaas: ' .
                        ($msgApi ?: 'Falha ao criar cobrança no Asaas.') .
                        '<br><br><small>' . $debug . '</small>';

            erro_pagamento($msgFinal);
        }

                $paymentId   = $dataPay['id'];
        $invoiceUrl  = $dataPay['invoiceUrl'] ?? null;
        $statusRaw   = $dataPay['status'] ?? null;   // ex: PENDING, CONFIRMED, RECEIVED...
        $statusLocal = $statusRaw ?: 'pending';

        if (!$invoiceUrl) {
            $debug = 'URL: ' . $urlPay .
                     ' | Resposta Asaas (payments): ' . htmlspecialchars($respPay);

            erro_pagamento(
                'Cobrança criada no Asaas, mas não retornou invoiceUrl para redirecionar.' .
                '<br><br><small>' . $debug . '</small>'
            );
        }

        // 3.b) Salva/atualiza a venda localmente para o e-mail do usuário
        try {
            $stmt = $pdo->prepare("
              INSERT INTO vendas_app (
                gateway_slug,
                payment_id, preference_id, status, status_detail,
                nome, email, whatsapp, tipo, doc, amount, raw_json
              )
              VALUES (
                :gateway_slug,
                :payment_id, :preference_id, :status, :status_detail,
                :nome, :email, :whatsapp, :tipo, :doc, :amount, :raw_json
              )
              ON DUPLICATE KEY UPDATE
                nome       = VALUES(nome),
                email      = VALUES(email),
                whatsapp   = VALUES(whatsapp),
                tipo       = VALUES(tipo),
                doc        = VALUES(doc),
                amount     = VALUES(amount),
                status     = VALUES(status),
                status_detail = VALUES(status_detail),
                raw_json   = VALUES(raw_json),
                updated_at = NOW()
            ");

            $stmt->execute([
                ':gateway_slug'   => 'asaas',
                ':payment_id'     => $paymentId,
                ':preference_id'  => null,
                ':status'         => $statusLocal,
                ':status_detail'  => $statusRaw,
                ':nome'           => $nome,
                ':email'          => $email,
                ':whatsapp'       => $whats,
                ':tipo'           => $tipo,
                ':doc'            => $doc,
                ':amount'         => $preco,
                ':raw_json'       => $respPay,
            ]);
        } catch (Throwable $e) {
            // Não vamos quebrar o redirecionamento por causa disso,
            // mas em produção você pode logar esse erro.
            // file_put_contents(__DIR__.'/logs_asaas_insert.txt', $e->getMessage().PHP_EOL, FILE_APPEND);
        }

        // 4) Redireciona o usuário para o invoiceUrl (tela de pagamento Asaas)
        header('Location: ' . $invoiceUrl);
        exit;


    // =================================================
    // QUALQUER OUTRO SLUG INVÁLIDO
    // =================================================
    default:
        erro_pagamento('Gateway de pagamento padrão inválido. Verifique as configurações em Admin → Gateways.');
}
