<?php
// functions.php
// Funções auxiliares para configs e gateways

function get_app_setting(PDO $pdo, string $key, $default = null): ?string {
    $stmt = $pdo->prepare("SELECT `value` FROM app_settings WHERE `key` = :key LIMIT 1");
    $stmt->execute([':key' => $key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row && $row['value'] !== null) {
        return $row['value'];
    }
    return $default;
}

function set_app_setting(PDO $pdo, string $key, $value): void {
    $stmt = $pdo->prepare("
        INSERT INTO app_settings (`key`, `value`)
        VALUES (:key, :value)
        ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)
    ");
    $stmt->execute([':key' => $key, ':value' => $value]);
}

function get_all_app_settings(PDO $pdo): array {
    $stmt = $pdo->query("SELECT `key`, `value` FROM app_settings");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $out = [];
    foreach ($rows as $r) {
        $out[$r['key']] = $r['value'];
    }
    return $out;
}

function hydrate_gateway_row(array $gw): array {
    $gw['config'] = [];
    if (!empty($gw['config_json'])) {
        $cfg = json_decode($gw['config_json'], true);
        if (is_array($cfg)) {
            $gw['config'] = $cfg;
        }
    }
    return $gw;
}

function get_gateways(PDO $pdo): array {
    $stmt = $pdo->query("SELECT * FROM payment_gateways ORDER BY id ASC");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($rows as &$r) {
        $r = hydrate_gateway_row($r);
    }
    return $rows;
}

function get_gateway_by_id(PDO $pdo, int $id): ?array {
    $stmt = $pdo->prepare("SELECT * FROM payment_gateways WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $id]);
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function get_gateway_by_slug(PDO $pdo, string $slug): ?array {
    $stmt = $pdo->prepare("SELECT * FROM payment_gateways WHERE slug = :slug LIMIT 1");
    $stmt->execute([':slug' => $slug]);
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function get_default_gateway(PDO $pdo): ?array {
    $stmt = $pdo->query("SELECT * FROM payment_gateways WHERE is_default = 1 AND is_active = 1 LIMIT 1");
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) {
        $stmt = $pdo->query("SELECT * FROM payment_gateways WHERE is_active = 1 LIMIT 1");
        $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function set_default_gateway(PDO $pdo, int $id): void {
    $pdo->beginTransaction();
    try {
        $pdo->exec("UPDATE payment_gateways SET is_default = 0");
        $stmt = $pdo->prepare("UPDATE payment_gateways SET is_default = 1 WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
}

// Verifica se o e-mail tem ao menos um pagamento aprovado em qualquer gateway
function usuario_tem_pagamento_aprovado(PDO $pdo, string $email): bool {
    $stmt = $pdo->prepare("
        SELECT 1
        FROM vendas_app
        WHERE email = :email
          AND status = 'approved'
        LIMIT 1
    ");
    $stmt->execute([':email' => $email]);
    return (bool)$stmt->fetchColumn();
}
