<?php
// login.php

require __DIR__ . '/config.php';
require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
  session_start();
}

$pageTitle = 'Login - Formação de Preço de Venda';

$erro = '';
$info = 'Informe o mesmo e-mail que você usou na hora do pagamento ou no Trial.';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $email = isset($_POST['email']) ? trim($_POST['email']) : '';

  if (empty($email)) {
    $erro = 'Informe o e-mail utilizado na compra ou no Trial.';
  } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $erro = 'E-mail inválido.';
  } else {

    // 1) TENTA LOGIN COMO CLIENTE PAGANTE (vendas_app)
    $sqlVenda = "
      SELECT *
      FROM vendas_app
      WHERE email = :email
        AND status = 'approved'
      ORDER BY id DESC
      LIMIT 1
    ";
    $stmtVenda = $pdo->prepare($sqlVenda);
    $stmtVenda->execute([':email' => $email]);
    $rowVenda = $stmtVenda->fetch();

    if ($rowVenda) {
      // Cliente com licença completa
      $_SESSION['usuario_autorizado'] = true;
      $_SESSION['usuario_email']      = $rowVenda['email'];
      $_SESSION['usuario_nome']       = $rowVenda['nome'] ?: $rowVenda['email'];
      $_SESSION['usuario_payment_id'] = $rowVenda['payment_id'];
      $_SESSION['usuario_trial']      = false;
      unset($_SESSION['trial_restantes']);

      header('Location: index.php');
      exit;
    }

    // 2) SE NÃO TEM VENDA, VERIFICA SE É USUÁRIO TRIAL
    $sqlTrial = "
      SELECT *,
             TIMESTAMPDIFF(DAY, created_at, NOW()) AS dias_desde
      FROM usuarios_trial
      WHERE email = :email
      LIMIT 1
    ";
    $stmtTrial = $pdo->prepare($sqlTrial);
    $stmtTrial->execute([':email' => $email]);
    $rowTrial = $stmtTrial->fetch();

    if ($rowTrial) {
      $simCount   = (int)$rowTrial['sim_count'];
      $diasDesde  = isset($rowTrial['dias_desde']) ? (int)$rowTrial['dias_desde'] : 0;

      $limiteSims = (int) get_app_setting($pdo, 'trial_max_simulacoes', '3');
	  $limiteDias = (int) get_app_setting($pdo, 'trial_max_dias', '7');

      $trialExpirado = ($simCount >= $limiteSims) || ($diasDesde >= $limiteDias);

      if ($trialExpirado) {
        // TRIAL EXPIRADO → MENSAGEM + REDIRECIONA PARA PAGAMENTO
        $_SESSION['mensagem_trial_expirado'] =
          "Seu período de teste gratuito já foi utilizado ou expirou. " .
          "Para continuar usando o aplicativo, conclua a compra da licença.";

        $nomeUrl  = urlencode($rowTrial['nome']);
        $emailUrl = urlencode($rowTrial['email']);

        header("Location: pagamento.php?trial=1&nome={$nomeUrl}&email={$emailUrl}");
        exit;
      } else {
        // TRIAL AINDA ATIVO → ENTRA EM MODO TRIAL
        $_SESSION['usuario_autorizado'] = true;
        $_SESSION['usuario_email']      = $rowTrial['email'];
        $_SESSION['usuario_nome']       = $rowTrial['nome'];
        $_SESSION['usuario_trial']      = true;

        $restantes = max(0, $limiteSims - $simCount);
        $_SESSION['trial_restantes'] = $restantes;

        header('Location: index.php');
        exit;
      }
    }

    // 3) NÃO TEM VENDA NEM TRIAL
    $erro = 'Não encontramos licença ativa ou período de teste para este e-mail.
Você pode ativar um Trial (se ainda não usou) ou realizar a compra da licença.';
  }
}

include __DIR__ . '/header.php';
?>

<div class="grid">
  <div class="card" style="max-width:520px;margin:0 auto;">
    <div class="card-header">
      <div class="card-title">Login de acesso ao APP</div>
      <span class="badge">CLIENTE / TRIAL</span>
    </div>

    <div class="field">
      <label>E-mail utilizado na compra ou Trial</label>
      <input
        type="email"
        name="email"
        id="email"
        form="login_form"
        placeholder="voce@empresa.com"
        value="<?php echo isset($email) ? htmlspecialchars($email) : ''; ?>"
      >
      <div class="hint">
        <?php echo htmlspecialchars($info); ?>
      </div>
    </div>

    <?php if (!empty($erro)): ?>
      <div class="metric metric-danger" style="margin-top:8px;">
        <div class="metric-label">
          <span>Não foi possível entrar</span>
        </div>
        <div class="metric-value metric-small" style="font-size:0.8rem;">
          <?php echo nl2br(htmlspecialchars($erro)); ?>
        </div>
      </div>
    <?php endif; ?>

    <hr class="divider">

    <form id="login_form" method="post" action="login.php" style="margin:0;padding:0;">
      <div class="field">
        <button
          type="submit"
          class="btn"
          style="width:100%;justify-content:center;font-size:0.9rem;"
        >
          <span class="dot"></span> Entrar no aplicativo
        </button>
      </div>
    </form>

    <div class="field" style="margin-top:10px;">
      <div class="hint">
        Ainda não comprou o APP?<br>
        <a href="cadastro.php" style="color:#38bdf8;text-decoration:none;">
          Clique aqui para se cadastrar e realizar o pagamento.
        </a>
      </div>
    </div>

    <div class="field" style="margin-top:6px;">
      <div class="hint">
        Quer testar antes de comprar?<br>
        <a href="trial.php" style="color:#a5b4fc;text-decoration:none;">
          Acessar versão Trial (3 simulações grátis)
        </a>
      </div>
    </div>
  </div>
</div>

<?php include __DIR__ . '/footer.php'; ?>
