<?php
// webhook_pagarme.php
// Notificações (postbacks/webhooks) da Pagar.me

require __DIR__ . '/config.php';
require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$LOG_WEBHOOK = false;

function log_webhook_pagarme($msg) {
    global $LOG_WEBHOOK;
    if (!$LOG_WEBHOOK) return;
    $file = __DIR__ . '/logs_webhook_pagarme.txt';
    file_put_contents($file, '['.date('Y-m-d H:i:s')."] $msg\n", FILE_APPEND);
}

// Carrega gateway 'pagarme'
$gw = get_gateway_by_slug($pdo, 'pagarme');
if (!$gw) {
    http_response_code(500);
    echo 'Gateway Pagar.me não configurado';
    exit;
}
$config = $gw['config'] ?? [];
$apiKey = $config['api_key'] ?? null;

$rawBody  = file_get_contents('php://input');
$headers  = getallheaders();
$signature = $headers['X-Hub-Signature'] ?? ($headers['x-hub-signature'] ?? null);

log_webhook_pagarme('==== NOVO POSTBACK PAGAR.ME ====');
log_webhook_pagarme('Headers: ' . json_encode($headers));
log_webhook_pagarme('Body: ' . $rawBody);

// Validação básica da assinatura, se apiKey e assinatura existirem
if ($apiKey && $signature) {
    // Pagar.me usa HMAC-SHA1(body, API_KEY) e envia em X-Hub-Signature
    $expected = 'sha1=' . hash_hmac('sha1', $rawBody, $apiKey);
    if (!hash_equals($expected, $signature)) {
        log_webhook_pagarme('Assinatura inválida. Esperado: '.$expected.' / Recebido: '.$signature);
        http_response_code(401);
        echo 'Invalid signature';
        exit;
    }
}

$data = json_decode($rawBody, true);
if (!is_array($data)) {
    http_response_code(400);
    echo 'JSON inválido';
    exit;
}

// Estrutura típica: { "type": "order.paid", "data": { ... } } 
$type      = $data['type'] ?? null;
$dataInner = $data['data'] ?? [];

$externalId = $dataInner['id'] ?? null;      // id da order/transaction
$statusRaw  = $dataInner['status'] ?? null;  // status da order/transaction
$amountCents= $dataInner['amount'] ?? null;  // geralmente em centavos
$amount     = $amountCents !== null ? ($amountCents / 100) : null;

$customer   = $dataInner['customer'] ?? [];
$email      = $customer['email'] ?? null;
$nome       = $customer['name']  ?? null;

// Normaliza status: se for "paid", tratamos como "approved"
$status = $statusRaw;
if ($statusRaw === 'paid') {
    $status = 'approved';
}

// Só vale a pena gravar se tiver ID
if (!$externalId) {
    log_webhook_pagarme('Sem id em data.data.id');
    http_response_code(400);
    echo 'Sem id de pagamento';
    exit;
}

try {
    $stmt = $pdo->prepare("
      INSERT INTO vendas_app (
        gateway_slug,
        payment_id, preference_id, status, status_detail,
        nome, email, whatsapp, tipo, doc, amount, raw_json
      )
      VALUES (
        :gateway_slug,
        :payment_id, :preference_id, :status, :status_detail,
        :nome, :email, :whatsapp, :tipo, :doc, :amount, :raw_json
      )
      ON DUPLICATE KEY UPDATE
        status = VALUES(status),
        status_detail = VALUES(status_detail),
        amount = VALUES(amount),
        raw_json = VALUES(raw_json),
        updated_at = NOW()
    ");

    $stmt->execute([
      ':gateway_slug'  => 'pagarme',
      ':payment_id'    => $externalId,
      ':preference_id' => null,
      ':status'        => $status,
      ':status_detail' => $statusRaw,
      ':nome'          => $nome,
      ':email'         => $email,
      ':whatsapp'      => null,
      ':tipo'          => null,
      ':doc'           => null,
      ':amount'        => $amount,
      ':raw_json'      => $rawBody,
    ]);

    log_webhook_pagarme("vendas_app atualizado para pagarme/{$externalId}, status={$status}");

    http_response_code(200);
    echo 'OK';

} catch (Throwable $e) {
    log_webhook_pagarme('Erro: ' . $e->getMessage());
    http_response_code(500);
    echo 'Erro interno';
}
