<?php
// webhook_asaas.php
// Webhooks do Asaas: atualiza status da venda na tabela vendas_app
require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

$LOG_WEBHOOK = false;

function log_webhook_asaas($msg) {
    global $LOG_WEBHOOK;
    if (!$LOG_WEBHOOK) return;
    $file = __DIR__ . '/logs_webhook_asaas.txt';
    file_put_contents($file, '['.date('Y-m-d H:i:s')."] $msg\n", FILE_APPEND);
}

// Carrega gateway 'asaas'
$gw = get_gateway_by_slug($pdo, 'asaas');
if (!$gw) {
    http_response_code(500);
    echo 'Gateway Asaas não configurado';
    exit;
}
$config = $gw['config'] ?? [];

// Authentication Token configurado no painel do Asaas
$webhookSecret = $config['webhook_secret'] ?? null;

$rawBody = file_get_contents('php://input');
$headers = getallheaders();

log_webhook_asaas('==== NOVO WEBHOOK ASAAS ====');
log_webhook_asaas('Headers: ' . json_encode($headers));
log_webhook_asaas('Body: ' . $rawBody);

// Validação do header asaas-access-token se tiver secret configurado
$tokenHeader = $headers['asaas-access-token'] ?? ($headers['Asaas-Access-Token'] ?? null);
if ($webhookSecret && $tokenHeader) {
    if (!hash_equals($webhookSecret, $tokenHeader)) {
        log_webhook_asaas('Token inválido. Esperado: '.$webhookSecret.' / Recebido: '.$tokenHeader);
        http_response_code(401);
        echo 'Invalid token';
        exit;
    }
}

$data = json_decode($rawBody, true);
if (!is_array($data)) {
    http_response_code(400);
    echo 'JSON inválido';
    exit;
}

// Estrutura típica: { "event": "PAYMENT_RECEIVED", "payment": { ... } }
$event   = $data['event']   ?? null;
$payment = $data['payment'] ?? null;

if (!$payment || !is_array($payment)) {
    log_webhook_asaas('Webhook sem objeto payment');
    http_response_code(400);
    echo 'Sem payment';
    exit;
}

$paymentId  = $payment['id']     ?? null;
$statusRaw  = $payment['status'] ?? null;
$amount     = $payment['value']  ?? null;

if (!$paymentId) {
    http_response_code(400);
    echo 'payment.id não informado';
    exit;
}

// Normalização de status Asaas -> nosso 'approved'
$status = $statusRaw;
$aprovados = ['RECEIVED', 'CONFIRMED', 'RECEIVED_IN_CASH', 'RECEIVED_PIX'];
if ($statusRaw && in_array($statusRaw, $aprovados, true)) {
    $status = 'approved';
}

try {
    // Primeiro, tenta atualizar um registro existente (criado no momento da compra)
    $stmt = $pdo->prepare("
      UPDATE vendas_app
      SET status = :status,
          status_detail = :status_detail,
          amount = :amount,
          raw_json = :raw_json,
          updated_at = NOW()
      WHERE gateway_slug = 'asaas'
        AND payment_id = :payment_id
    ");

    $stmt->execute([
        ':status'        => $status,
        ':status_detail' => $statusRaw,
        ':amount'        => $amount,
        ':raw_json'      => $rawBody,
        ':payment_id'    => $paymentId,
    ]);

    if ($stmt->rowCount() === 0) {
        // Se por algum motivo não existir (caso raro), cria um novo registro "mudo"
        $stmtIns = $pdo->prepare("
          INSERT INTO vendas_app (
            gateway_slug,
            payment_id, preference_id, status, status_detail,
            nome, email, whatsapp, tipo, doc, amount, raw_json
          )
          VALUES (
            :gateway_slug,
            :payment_id, :preference_id, :status, :status_detail,
            NULL, NULL, NULL, NULL, NULL, :amount, :raw_json
          )
        ");
        $stmtIns->execute([
            ':gateway_slug'  => 'asaas',
            ':payment_id'    => $paymentId,
            ':preference_id' => null,
            ':status'        => $status,
            ':status_detail' => $statusRaw,
            ':amount'        => $amount,
            ':raw_json'      => $rawBody,
        ]);
        log_webhook_asaas("vendas_app inserido para asaas/{$paymentId}, status={$status}");
    } else {
        log_webhook_asaas("vendas_app atualizado para asaas/{$paymentId}, status={$status}");
    }

    http_response_code(200);
    echo 'OK';

} catch (Throwable $e) {
    log_webhook_asaas('Erro: ' . $e->getMessage());
    http_response_code(500);
    echo 'Erro interno';
}
