<?php
// webhook_mp.php
// Endpoint para receber notificações Webhook do Mercado Pago

require __DIR__ . '/vendor/autoload.php';
require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

use MercadoPago\MercadoPagoConfig;
use MercadoPago\Client\Payment\PaymentClient;
use MercadoPago\Exceptions\MPApiException;

// Ative se quiser logar em arquivo
$LOG_WEBHOOK = false;

function log_webhook_mp($msg) {
    global $LOG_WEBHOOK;
    if (!$LOG_WEBHOOK) return;
    $file = __DIR__ . '/logs_webhook_mp.txt';
    file_put_contents($file, '['.date('Y-m-d H:i:s')."] $msg\n", FILE_APPEND);
}

// Carrega config do gateway 'mercadopago' pelo BD
$gw = get_gateway_by_slug($pdo, 'mercadopago');
$config = $gw['config'] ?? [];

$accessToken = $config['access_token'] ?? ($MP_ACCESS_TOKEN ?? null);
if (!$accessToken) {
    http_response_code(500);
    echo 'MP Access Token não configurado';
    exit;
}

MercadoPagoConfig::setAccessToken($accessToken);

// Corpo e headers
$rawBody = file_get_contents('php://input');
$headers = getallheaders();

log_webhook_mp('==== NOVA NOTIFICAÇÃO MP ====');
log_webhook_mp('Headers: ' . json_encode($headers));
log_webhook_mp('Body: ' . $rawBody);

// TODO: se quiser, implementar validação de assinatura com $config['webhook_secret']

$data = json_decode($rawBody, true);

// Encontra payment_id
$paymentId = null;
if (is_array($data)) {
    $type = $data['type']  ?? $data['action'] ?? null;
    if ($type === 'payment' && !empty($data['data']['id'])) {
        $paymentId = $data['data']['id'];
    }
}
if (!$paymentId && isset($_GET['id'])) {
    $topic = $_GET['topic'] ?? $_GET['type'] ?? null;
    if ($topic === 'payment') {
        $paymentId = $_GET['id'];
    }
}

if (!$paymentId) {
    log_webhook_mp('Nenhum payment_id encontrado.');
    http_response_code(400);
    echo 'payment_id não encontrado';
    exit;
}

log_webhook_mp("payment_id: {$paymentId}");

try {
    $client  = new PaymentClient();
    $payment = $client->get($paymentId);

    $statusApi    = $payment->status ?? null;
    $statusDetail = $payment->status_detail ?? null;
    $amount       = $payment->transaction_amount ?? null;
    $metadata     = $payment->metadata ?? null;

    $nome  = $metadata->nome      ?? null;
    $email = $metadata->email     ?? ($payment->payer->email ?? null);
    $whats = $metadata->whatsapp  ?? null;
    $tipo  = $metadata->tipo      ?? null;
    $doc   = $metadata->doc       ?? null;

    $stmt = $pdo->prepare("
      INSERT INTO vendas_app (
        gateway_slug,
        payment_id, preference_id, status, status_detail,
        nome, email, whatsapp, tipo, doc, amount, raw_json
      )
      VALUES (
        :gateway_slug,
        :payment_id, :preference_id, :status, :status_detail,
        :nome, :email, :whatsapp, :tipo, :doc, :amount, :raw_json
      )
      ON DUPLICATE KEY UPDATE
        status = VALUES(status),
        status_detail = VALUES(status_detail),
        amount = VALUES(amount),
        raw_json = VALUES(raw_json),
        updated_at = NOW()
    ");

    $stmt->execute([
      ':gateway_slug'  => 'mercadopago',
      ':payment_id'    => $paymentId,
      ':preference_id' => $payment->order->id ?? null,
      ':status'        => $statusApi,
      ':status_detail' => $statusDetail,
      ':nome'          => $nome,
      ':email'         => $email,
      ':whatsapp'      => $whats,
      ':tipo'          => $tipo,
      ':doc'           => $doc,
      ':amount'        => $amount,
      ':raw_json'      => json_encode($payment, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES),
    ]);

    log_webhook_mp("vendas_app atualizado para {$paymentId}, status={$statusApi}");

    http_response_code(200);
    echo 'OK';

} catch (MPApiException $e) {
    log_webhook_mp('MPApiException: ' . $e->getMessage());
    http_response_code(500);
    echo 'Erro ao consultar pagamento';
} catch (Exception $e) {
    log_webhook_mp('Exception: ' . $e->getMessage());
    http_response_code(500);
    echo 'Erro interno';
}
