require('dotenv').config();
const express = require('express');
const http = require('http');
const cors = require('cors');
const { Server } = require('socket.io');
const fs = require('fs');
const path = require('path');
const db = require('./config/db');

const authRoutes = require('./routes/auth');
const webhookRoutes = require('./routes/webhooks');
const conversationRoutes = require('./routes/conversations');
const { authMiddleware } = require('./middlewares/auth');

const app = express();
const server = http.createServer(app);

const io = new Server(server, {
  cors: {
    origin: '*'
  }
});

app.set('io', io);

app.use(cors());
app.use(express.json());

app.get('/', (req, res) => {
  res.json({ status: 'ok', message: 'WhatsApp Helpdesk API' });
});

app.use('/auth', authRoutes);
app.use('/webhooks', webhookRoutes);
app.use('/conversations', authMiddleware, conversationRoutes);

io.on('connection', socket => {
  console.log('Socket conectado', socket.id);

  socket.on('join_user', userId => {
    socket.join(`user:${userId}`);
  });

  socket.on('join_conversation', conversationId => {
    socket.join(`conversation:${conversationId}`);
  });
});

async function runMigrations() {
  try {
    const schemaPath = path.join(__dirname, 'sql', 'schema.sql');
    const sql = fs.readFileSync(schemaPath, 'utf8');
    const statements = sql
      .split(';')
      .map(s => s.trim())
      .filter(s => s.length);

    for (const stmt of statements) {
      await db.query(stmt);
    }
    console.log('Migrações executadas com sucesso');
  } catch (err) {
    console.error('Erro ao executar migrações:', err.message);
  }
}

const PORT = process.env.PORT || 3000;

server.listen(PORT, async () => {
  try {
    await db.query('SELECT 1');
    console.log('Banco conectado');
    await runMigrations();
  } catch (err) {
    console.error('Erro ao conectar no banco:', err.message);
  }
  console.log(`Backend rodando na porta ${PORT}`);
});
